/**
 * Klasse Karte
 * Bildelement, das beim Anklicken eine Textbeschreibung anzeigt
 *
 * @constructor
 * @param pPosX {float} Position.x
 * @param pPosY {float} Position.y
 * @param pWidth {float} Breite
 * @param pHeight {float} Höhe
 * @param pImgSrc {String} Bild-Pfad
 * @param pText {String} Kartentext
 */

function Card(pPosX, pPosY, pWidth, pHeight, pImgSrc, pText) {
    this.pType = "Card";
    this.pWidth = pWidth;
    this.pHeight = pHeight;
    this.pInitialWidth = pWidth;
    this.pInitialHeight = pHeight;
    this.pPosX = pPosX;
    this.pPosY = pPosY;
    this.pInitialPosX = pPosX;
    this.pInitialPosY = pPosY;
    this.pSolved = true;
    this.pImgSrc = pImgSrc;
    this.pJsonId = newJsonId;

    /**
     * Setzt die Position der Karte
     * @param {float} pPosX X-Position
     * @param {float} pPosY Y-Position
     */
    this.setPos = function (pPosX, pPosY) {
        this.pPosX = pPosX;
        this.pPosY = pPosY;
    };

    /**
     * Funktion wenn die Karte angeklickt wird
     */
    this.click = function () {
        if (this.pSolved) {
            $(this.pText).hide();
        } else {
            $(this.pText).show();
        }
        this.pSolved = !this.pSolved;
        this.setScale(this.pScale);
    };

    /**
     * Erstellt das Element
     * @param {jqueryelement} rParent
     */
    this.create = function (rParent) {
        this.rPatent = rParent;
        this.rSelector = $('<div class="card"></div>').appendTo(this.rPatent);
        $(this.rSelector).css("left", this.pPosX + "px");
        $(this.rSelector).css("top", this.pPosY + "px");
        $(this.rSelector).css("width", this.pWidth + "px");
        $(this.rSelector).css("height", this.pHeight + "px");

        this.rImg = $('<img src="' + this.pImgSrc + '" class="cardImg">').prependTo(this.rSelector);

        this.pText = $('<div class="cardText"><span>' + pText + "</span></div>").appendTo(this.rSelector);
        makeUnselectable(this.pText);

        var t = this;

        $(this.rSelector).click(function (e) {
            t.click();
        });

        this.click();
    };

    /**
     * Setzt die Skalierung
     * @param {float} pScale
     */
    this.setScale = function (pScale) {
        this.pScale = pScale;
        this.pWidth = this.pInitialWidth * this.pScale;
        this.pHeight = this.pInitialHeight * this.pScale;
        $(this.rSelector).css("width", this.pWidth + "px");
        $(this.rSelector).css("height", this.pHeight + "px");

        this.pPosX = this.pInitialPosX * this.pScale;
        this.pPosY = this.pInitialPosY * this.pScale;
        $(this.rSelector).css("left", this.pPosX + "px");
        $(this.rSelector).css("top", this.pPosY + "px");

        if (this.pSolved) {
            var fs = parseFloat($(this.pText).find("span").css("font-size")) * (parseFloat($(this.pText).css("width")) / parseFloat($(this.pText).find("span").css("width")));
            if (this.pMaxFontSize != null) {
                fs = Math.min(this.pMaxFontSize * pScale, fs);
            }
            if (this.pMinFontSize != null) {
                fs = Math.max(this.pMinFontSize * pScale, fs);
            }
            $(this.pText).find("span").css("font-size", fs);
            $(this.pText)
                .find("span")
                .css("top", (parseFloat($(this.pText).css("height")) - parseFloat($(this.pText).find("span").css("height"))) / 2);
            $(this.pText)
                .find("span")
                .css("left", (parseFloat($(this.pText).css("width")) - parseFloat($(this.pText).find("span").css("width"))) / 2);
        }
    };

    /**
     * Stellt die Farbe der Karte ein
     * @param {string rgb (r,g,b)} pColor
     */
    this.setBackgroundColor = function (pColor) {
        this.pBackgroundColor = pColor;
        $(this.pText).css("background-color", pColor);
    };

    /**
     * Stellt den Rand für die Hintergrundfarbe ein
     * @param {float} pLeft
     * @param {float} pRight
     * @param {float} pTop
     * @param {float} pBottom
     **/
    this.setBackgroundMargin = function (pLeft, pRight, pTop, pBottom) {
        $(this.pText).css("left", (pLeft / this.pInitialWidth) * 100 + "%");
        $(this.pText).css("width", ((this.pInitialWidth - pLeft - pRight) / this.pInitialWidth) * 100 + "%");
        $(this.pText).css("top", (pTop / this.pInitialHeight) * 100 + "%");
        $(this.pText).css("height", ((this.pInitialHeight - pTop - pBottom) / this.pInitialHeight) * 100 + "%");
    };

    /**
     * Stellt die Rundheit der Ecken der Hintergrundfarbe ein
     * @param {float} pRoundness
     **/
    this.setBackgroundRoundness = function (pRoundness) {
        $(this.pText).css("border-radius", pRoundness + "%");
    };

    /**
     * Setzt das Maximum für eine Dynamische Schriftgröße
     * @param {float} pSize
     */
    this.setMaxFontSize = function (pSize) {
        this.pMaxFontSize = pSize;
    };

    /**
     * Setzt das Minimum für eine Dynamische Schriftgröße
     * @param {float} pSize
     */
    this.setMinFontSize = function (pSize) {
        this.pMinFontSize = pSize;
    };

    /**
     * Stellt den Loesungszustand her
     */
    this.solve = function () {
        if (!this.pSolved) {
            this.click();
        }
    };

    /**
     * Gibt an ob der Lösungszustand erreicht ist
     * @returns {boolean}
     */
    this.isSolved = function () {
        return this.pSolved;
    };

    /**
     * Stellt den Ausganszustand her
     */
    this.reset = function () {
        if (this.pSolved) {
            this.click();
        }
    };

    /**
     * Speichert den Zustand in einem JSON-Objekt
     * @returns {JSON} Zustand
     */
    this.export = function () {
        var obj = {
            pJsonId: this.pJsonId,
            pType: this.pType,
            pSolved: this.pSolved,
        };

        return obj;
    };

    /**
     * Setzt den in einem JSON-Objekt gespeicherten Zustand um
     * @param {JSON} rObj
     */
    this.import = function (rObj) {
        if (rObj.pType == "Card") {
            if (this.isSolved() != rObj.pSolved) {
                this.click();
            }
        }
    };
}
